// This file is part of retro-gtk. License: GPL-3.0+.

/**
 * SECTION:retro-pixbuf
 * @short_description: Functions for #GdkPixbuf
 * @title: Pixbuf Functions
 * @See_also: #GdkPixbuf, #RetroCoreView
 *
 * Convenience functions to handle the options of a #GdkPixbuf.
 */

#include "retro-pixbuf.h"
#include <string.h>

/**
 * retro_pixbuf_get_aspect_ratio:
 * @pixbuf: a #GdkPixbuf
 *
 * Gets the aspect ratio of @pixbuf by reading the 'aspect-ratio' pixbuf option.
 *
 * Returns: the aspect ratio, or 0 if the option is not set or its value is invalid.
 */
gfloat
retro_pixbuf_get_aspect_ratio (GdkPixbuf *pixbuf)
{
  const gchar *aspect_ratio_str;
  gfloat result = 0.f;

  g_return_val_if_fail (GDK_IS_PIXBUF (pixbuf), 0.f);

  aspect_ratio_str = gdk_pixbuf_get_option (pixbuf, "aspect-ratio");

  if (!aspect_ratio_str)
    return 0.f;

  sscanf (aspect_ratio_str, "%g", &result);

  return result;
}

/**
 * retro_pixbuf_set_aspect_ratio:
 * @pixbuf: a #GdkPixbuf
 * @aspect_ratio: the aspect ratio value
 *
 * Sets the aspect ratio of @pixbuf by setting the 'aspect-ratio' pixbuf option.
 * Use retro_pixbuf_get_aspect_ratio() to retrieve it.
 */
void
retro_pixbuf_set_aspect_ratio (GdkPixbuf *pixbuf,
                               gfloat     aspect_ratio)
{
  g_autofree gchar *aspect_ratio_string = NULL;

  g_return_if_fail (GDK_IS_PIXBUF (pixbuf));
  g_return_if_fail (aspect_ratio > 0.f);

  aspect_ratio_string = g_strdup_printf ("%g", aspect_ratio);

  gdk_pixbuf_remove_option (pixbuf, "aspect-ratio");

  gdk_pixbuf_set_option (pixbuf, "aspect-ratio", aspect_ratio_string);
}
