#!/usr/bin/env python3
# -*- Mode: python; indent-tabs-mode: nil; tab-width: 40 -*-
# vim: set filetype=python:
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

import os
import json


def gen_includes(export_prefix, path_prefix):
    result = ""

    exports = []
    for f in os.listdir(path_prefix):
        # Explicitly ignore the "{gtest,gmock}/internal/custom" paths, as we
        # replace files in them for custom configurations.
        if f == "custom":
            continue

        path = os.path.join(path_prefix, f)
        if os.path.isfile(path):
            if os.path.splitext(f)[1] != ".h":
                continue
            exports.append(path)
        else:
            result += gen_includes(export_prefix + "." + f, path)

    if exports:
        result += "%s += [\n" % export_prefix
        for export in sorted(exports):
            result += "    %s,\n" % json.dumps(export)
        result += "]\n"

    return result


if __name__ == "__main__":
    GTEST_PATH = "googletest/include/gtest"
    GMOCK_PATH = "googlemock/include/gmock"

    exports = "\n".join(
        [
            # Normal include paths used by most code
            gen_includes("EXPORTS.gtest", GTEST_PATH),
            gen_includes("EXPORTS.gmock", GMOCK_PATH),
            # webrtc.org unit tests use this include path
            gen_includes("EXPORTS.testing.gtest.include.gtest", GTEST_PATH),
            gen_includes("EXPORTS.testing.gmock.include.gmock", GMOCK_PATH),
        ]
    )
    exports = exports.replace("\n", "\n    ")

    mozbuild = f"""\
# -*- Mode: python; indent-tabs-mode: nil; tab-width: 40 -*-
# vim: set filetype=python:
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# !!! THIS FILE IS AUTOGENERATED USING gen_moz_build.py DO NOT EDIT !!!

with Files("**"):
    BUG_COMPONENT = ("Testing", "GTest")
    SCHEDULES.exclusive = ["gtest"]

if CONFIG["ENABLE_TESTS"]:

    {exports}

    SOURCES += [
        "googlemock/src/gmock-all.cc",
        "googletest/src/gtest-all.cc",
    ]

    Library("gtest")

    LOCAL_INCLUDES += [
        "googlemock",
        "googletest",
    ]

    if CONFIG["OS_ARCH"] == "WINNT":
        DEFINES["UNICODE"] = True

    FINAL_LIBRARY = "xul-gtest"
"""

    with open("moz.build", "w") as f:
        f.write(mozbuild)
